﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Text;
using System.Data;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;

namespace VeteransAffairs.Registries.BusinessManager.TBI
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class PolyTraumaPatientManager : BaseBO
    {
        private UserAccountManager _user = (UserAccountManager)System.Threading.Thread.CurrentPrincipal;
        private PolyTraumaEntriesManager polyTraumaEntriesManager = new PolyTraumaEntriesManager();

        public PolyTraumaPatientManager() 
        {
            _defaultSortField = "sNum";
            
        }

        private IQueryable<TBI_POLYTRAUMA> LinqAll(RegistriesDataAccess _db)
        {
            //populate LinqAll

            //polytrauma does not have a seperate list for patients, therefore use the PolyTraumaEntriesManager to return
            //an IQueryable of poly trauma entries that the current user has permission to access.

            IQueryable<TBI_POLYTRAUMA> tempLinq = polyTraumaEntriesManager.SelectIQueryable(_db);

            return tempLinq;

        }

        private void SetLoadWithForListViews(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.STD_INSTITUTION);  //loads Institution for PATIENT_HOME_VAMC_ID
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.STD_INSTITUTION1); //loads Institution for POLYTRAUMA_CENTER_ID
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_INSTITUTION_PARENT); //loads parent institution of any institution in order to get the VISN
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.STD_TBIPROVIDERLOC);
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.STD_TBIFACILITYSRC);
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.PATIENT);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }
        
        #region Method for Check Existence
        #endregion

        #region Methods for Select

        public IEnumerable<TBI_POLYTRAUMA> Select(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            using (_db = GetDataContext())
            {
                List<TBI_POLYTRAUMA> entities;
                entities = SelectLinqFilter(_db).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }

        public int SelectCount(string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                return SelectLinqFilter(_db).Count();
            }
        }

        private IQueryable<TBI_POLYTRAUMA> SelectLinqFilter(RegistriesDataAccess _db)
        {
            IQueryable<TBI_POLYTRAUMA> linqFilter = LinqAll(_db);
            return linqFilter;
        }

        #endregion

        #region Methods for SelectBySSNSearch

        public IEnumerable<TBI_POLYTRAUMA> SelectCriteriaSearch(string fromDate, string toDate, string visn, string snum, string lastName,
            string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;

            }

            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                IEnumerable<TBI_POLYTRAUMA> entities;
                entities = SelectCriteriaSearchLinqFilter(_db, fromDate, toDate, visn, snum, lastName).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }

        }

        public int SelectCriteriaSearchCount(string fromDate, string toDate, string visn, string snum, string lastName, string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                return SelectCriteriaSearchLinqFilter(_db, fromDate, toDate, visn, snum, lastName).Count();
            }
        }

        private IQueryable<TBI_POLYTRAUMA> SelectCriteriaSearchLinqFilter(RegistriesDataAccess _db, string startDate, string endDate, string visn, string ssn, string lastName)
        {
            IQueryable<TBI_POLYTRAUMA> linqFilter = LinqAll(_db);

            List<Expression<Func<TBI_POLYTRAUMA, bool>>> whereList = new List<Expression<Func<TBI_POLYTRAUMA, bool>>>();
            //build ssn filter 
            Expression<Func<TBI_POLYTRAUMA, bool>> ssnExpression;

            if (String.IsNullOrEmpty(ssn))
            {
                ssnExpression = s => true;
            }
            else
            {
                ssnExpression = s => s.Snum.Replace("-", "").Replace(" ", "") == ssn.Replace("-", "").Replace(" ", "");
            }

            whereList.Add(ssnExpression);
            //last name filter 
            Expression<Func<TBI_POLYTRAUMA, bool>> lnExpression;

            if (String.IsNullOrEmpty(lastName))
            {
                lnExpression = s => true;
            }
            else
            {
                lnExpression = s => s.LAST_NAME.Contains(lastName);
            }

            whereList.Add(lnExpression);

            //referral start date
            Expression<Func<TBI_POLYTRAUMA, bool>> startDateExpression;

            if (String.IsNullOrEmpty(startDate))
            {
                startDateExpression = s => true;
            }
            else
            {
                startDateExpression = s => s.TBI_INJURY_DATE >= DateTime.Parse(startDate);
            }

            whereList.Add(startDateExpression);

            //referral end date
            Expression<Func<TBI_POLYTRAUMA, bool>> endDateExpression;

            if (String.IsNullOrEmpty(endDate))
            {
                endDateExpression = s => true;
            }
            else
            {
                endDateExpression = s => s.TBI_INJURY_DATE <= DateTime.Parse(endDate);
            }

            whereList.Add(endDateExpression);

            //visn number
            Expression<Func<TBI_POLYTRAUMA, bool>> visnExpression;

            if (String.IsNullOrEmpty(visn) || int.Parse(visn) == 0)
            {
                visnExpression = s => true;
            }
            else
            {
                visnExpression = s => s.POLYTRAUMA_CENTER_ID == int.Parse(visn);
            }

            whereList.Add(visnExpression);

            foreach (var condition in whereList)
            {
                linqFilter = linqFilter.Where(condition);
            }

            return linqFilter;
        }

        #endregion
    }
}

